# Plugin for TWiki Enterprise Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2010-2019 Peter Thoeny, peter[at]thoeny.org
# and TWiki Contributors. All Rights Reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version. For
# more details read LICENSE in the root of this distribution.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# For licensing info read LICENSE file in the TWiki root.

=pod

---+ package GeoLookupPlugin

=cut

package TWiki::Plugins::GeoLookupPlugin;

# Always use strict to enforce variable scoping
use strict;

require TWiki::Func;    # The plugins API
require TWiki::Plugins; # For the API version

our $VERSION = '$Rev: 30651 (2019-05-04) $';
our $RELEASE = '2019-05-04';

our $SHORTDESCRIPTION = 'Lookup geolocation by IP address or domain name';
our $NO_PREFS_IN_TOPIC = 0;

# Name of this Plugin, only used in this module
my $pluginName = 'GeoLookupPlugin';

my $debug;
my $moduleEnum;
my $geoIP;
my $error;

=pod

---++ initPlugin($topic, $web, $user, $installWeb) -> $boolean
   * =$topic= - the name of the topic in the current CGI query
   * =$web= - the name of the web in the current CGI query
   * =$user= - the login name of the user
   * =$installWeb= - the name of the web the plugin is installed in

=cut

sub initPlugin {
    my( $topic, $web, $user, $installWeb ) = @_;

    # check for Plugins.pm versions
    if( $TWiki::Plugins::VERSION < 1.026 ) {
        TWiki::Func::writeWarning( "Version mismatch between $pluginName and Plugins.pm" );
        return 0;
    }

    # Get plugin settings
    $debug = TWiki::Func::getPreferencesFlag( "GEOLOOKUPPLUGIN_DEBUG" );

    $moduleEnum = 0;
    undef $geoIP;
    $error = '';

    TWiki::Func::registerTagHandler( 'GEOLOOKUP', \&_GEOLOOKUP );

    # Plugin correctly initialized
    return 1;
}

=pod

---++ _GEOLOOKUP()

This handles the %GEOLOOKUP{...}% variable

=cut

sub _GEOLOOKUP {
    my( $session, $params, $theTopic, $theWeb ) = @_;

    my $ip = $params->{_DEFAULT};
    my $text = $params->{format} || '$city, $region, $country_name';
    return( 'GEOLOOKUP error: IP address is missing' ) unless( $ip );

    _initGeoData() unless $moduleEnum;

    my $rec = _getGeoDataRecord( $ip );
    return( "GEOLOOKUP error: $error" ) unless $rec;

    $text =~ s/\$continent/$rec->{continent}||''/geo;
    $text =~ s/\$country_code/$rec->{country_code}||''/geo;
    $text =~ s/\$country_code3/$rec->{country_code3}||''/geo;
    $text =~ s/\$country_name/$rec->{country_name}||''/geo;
    $text =~ s/\$region/$rec->{region}||''/geo;
    $text =~ s/\$city/$rec->{city}||''/geo;
    $text =~ s/\$postal_code/$rec->{postal_code}||''/geo;
    $text =~ s/\$latitude/$rec->{latitude}||''/geo;
    $text =~ s/\$longitude/$rec->{longitude}||''/geo;
    $text =~ s/\$metro_code/$rec->{metro_code}||''/geo;
    $text =~ s/\$area_code/$rec->{area_code}||''/geo;
    if($text =~ /\$debug/) {
        use Data::Dumper;
        my $str = Dumper($rec);
        $str =~ s/\$VAR1 = //o;
        $str =~ s/         /  /go;
        $str =~ s/ +\};/\}/o;
        $text =~ s/\$debug/$str/geo;
    }
    return $text;
}

=pod

---++ _initGeoData()

Initialize the geo data.
Sets global $moduleEnum
   = -1 if not found,
   = 1 if Geo::IP
   = 2 if Geo::IP::NativePerl
   = 3 if MaxMind::DB::Reader using new GeoLite2-City.mmdb
Sets global $geoIP to module if found

=cut

sub _initGeoData {
    return unless( $moduleEnum == 0 );

    my $dataFile = $TWiki::cfg{GeoLookupPlugin}{GeoDataFile}
                || '/usr/share/GeoIP/GeoLite2-City.mmdb'
                || '/usr/local/share/GeoIP/GeoLiteCity.dat';

    if( eval "require MaxMind::DB::Reader" ) {
        $moduleEnum = 3;
        $geoIP = MaxMind::DB::Reader->new( file => $dataFile );
        $error = "Geo data file $dataFile not found" unless( $geoIP );

    } elsif( eval "require Geo::IP" ) {
        $moduleEnum = 1;
        $geoIP = Geo::IP->open( $dataFile, Geo::IP->GEOIP_STANDARD );
        $error = "Geo data file $dataFile not found" unless( $geoIP );

    } elsif( eval "require Geo::IP::PurePerl" ) {
        $moduleEnum = 2;
        eval {
            local $SIG{'__DIE__'};
            $geoIP = Geo::IP::PurePerl->open( $dataFile, Geo::IP::PurePerl->GEOIP_STANDARD );
        };
        $error = "Geo data file $dataFile not found" unless( $geoIP );

    } else {
        $moduleEnum = -1;
        undef $geoIP;
        $error = 'Module Geo::IP not found';
    }
}

=pod

---++ _getGeoDataRecord()

Get a geo data record

=cut

sub _getGeoDataRecord {
    my( $ip ) = @_;

    my $rec = undef;
    return $rec unless( $geoIP );

    unless( $ip =~ /^[0-9]+\.[0-9]/ ) { # FIXME: What about IPv6?
        $ip = _domainToIP( $ip );
    }

    $rec = {};
    $rec->{continent}     = '';
    $rec->{country_code}  = '';
    $rec->{country_code3} = '';
    $rec->{country_name}  = '';
    $rec->{region}        = '';
    $rec->{region_code}   = '';
    $rec->{city}          = '';
    $rec->{postal_code}   = '';
    $rec->{latitude}      = '';
    $rec->{longitude}     = '';
    $rec->{metro_code}    = '';
    $rec->{area_code}     = '';
    $rec->{time_zone}     = '';

    if( $moduleEnum == 1 ) {
        my $geoRec = $geoIP->record_by_addr( $ip );
        if( $geoRec ) {
            $rec->{country_code}  = $geoRec->country_code;
            $rec->{country_code3} = $geoRec->country_code3;
            $rec->{country_name}  = $geoRec->country_name;
            $rec->{region}        = $geoRec->region;
            $rec->{city}          = $geoRec->city;
            $rec->{postal_code}   = $geoRec->postal_code;
            $rec->{latitude}      = $geoRec->latitude;
            $rec->{longitude}     = $geoRec->longitude;
            $rec->{metro_code}    = $geoRec->metro_code;
            $rec->{area_code}     = $geoRec->area_code;
        }

    } elsif( $moduleEnum == 2 ) {
        # no need to convert domain name to IP address
        ( $rec->{country_code},
          $rec->{country_code3},
          $rec->{country_name},
          $rec->{region},
          $rec->{city},
          $rec->{postal_code},
          $rec->{latitude},
          $rec->{longitude},
          $rec->{metro_code},
          $rec->{area_code}
        ) = $geoIP->get_city_record( $ip );
        $rec->{continent }  = '';
        $rec->{region_code} = '';
        $rec->{time_zone}   = '';

    } elsif( $moduleEnum == 3 ) {
        my $geoRec = $geoIP->record_for_address( $ip );
        if( $geoRec ) {
            if( exists $geoRec->{continent} ) {
                $rec->{continent}  = $geoRec->{continent}->{names}->{en};
            }
            if( exists $geoRec->{country} && $geoRec->{country}->{iso_code} ) {
                $rec->{country_code}  = $geoRec->{country}->{iso_code};
                $rec->{country_code3} = _countryCodeToCode3( $rec->{country_code} );
            }
            if( exists $geoRec->{country} ) {
                $rec->{country_name}  = $geoRec->{country}->{names}->{en};
            }
            if( exists $geoRec->{subdivisions} && $geoRec->{subdivisions}[0] ) {
                $rec->{region}        = $geoRec->{subdivisions}[0]->{names}->{en};
                $rec->{region_code}   = $geoRec->{subdivisions}[0]->{iso_code};
            }
            if( exists $geoRec->{city} ) {
                $rec->{city}          = $geoRec->{city}->{names}->{en};
            }
            if( exists $geoRec->{postal} ) {
                $rec->{postal_code}   = $geoRec->{postal}->{code};
            }
            if( exists $geoRec->{location} && $geoRec->{location}->{latitude} ) {
                $rec->{latitude}      = $geoRec->{location}->{latitude};
            }
            if( exists $geoRec->{location} && $geoRec->{location}->{longitude} ) {
                $rec->{longitude}     = $geoRec->{location}->{longitude};
            }
            if( exists $geoRec->{location} && $geoRec->{location}->{time_zone} ) {
                $rec->{time_zone}     = $geoRec->{location}->{time_zone};
            }
            if( exists $geoRec->{location} && $geoRec->{location}->{metro_code} ) {
                $rec->{metro_code}    = $geoRec->{location}->{metro_code};
                $rec->{area_code}     = $geoRec->{location}->{metro_code};
            }
        }
    }
    return $rec;
}

=pod

---++ _domainToIP()

convert domain name to IP address

=cut

sub _domainToIP {
    my( $ip ) = @_;

    use Socket;
    my $packed_ip = gethostbyname( $ip );
    if (defined $packed_ip) {
        $ip = inet_ntoa($packed_ip);
    }
    return $ip;
}

=pod

---++ _fixCountryName()

=cut

sub _fixCountryName {
    my( $name ) = @_;
    $name = '' unless $name;
    $name =~ s/United States/USA/;
    return $name;
}

=pod

---++ _countryCodeToCode3()

=cut

sub _countryCodeToCode3 {
    my( $code ) = @_;
    my %codeMap = (
        AF => "AFG",
        AL => "ALB",
        DZ => "DZA",
        AS => "ASM",
        AD => "AND",
        AO => "AGO",
        AI => "AIA",
        AQ => "ATA",
        AG => "ATG",
        AR => "ARG",
        AM => "ARM",
        AW => "ABW",
        AU => "AUS",
        AT => "AUT",
        AZ => "AZE",
        BS => "BHS",
        BH => "BHR",
        BD => "BGD",
        BB => "BRB",
        BY => "BLR",
        BE => "BEL",
        BZ => "BLZ",
        BJ => "BEN",
        BM => "BMU",
        BT => "BTN",
        BO => "BOL",
        BO => "BOL",
        BA => "BIH",
        BW => "BWA",
        BV => "BVT",
        BR => "BRA",
        IO => "IOT",
        BN => "BRN",
        BN => "BRN",
        BG => "BGR",
        BF => "BFA",
        BI => "BDI",
        KH => "KHM",
        CM => "CMR",
        CA => "CAN",
        CV => "CPV",
        KY => "CYM",
        CF => "CAF",
        TD => "TCD",
        CL => "CHL",
        CN => "CHN",
        CX => "CXR",
        CC => "CCK",
        CO => "COL",
        KM => "COM",
        CG => "COG",
        CD => "COD",
        CK => "COK",
        CR => "CRI",
        CI => "CIV",
        CI => "CIV",
        HR => "HRV",
        CU => "CUB",
        CY => "CYP",
        CZ => "CZE",
        DK => "DNK",
        DJ => "DJI",
        DM => "DMA",
        DO => "DOM",
        EC => "ECU",
        EG => "EGY",
        SV => "SLV",
        GQ => "GNQ",
        ER => "ERI",
        EE => "EST",
        ET => "ETH",
        FK => "FLK",
        FO => "FRO",
        FJ => "FJI",
        FI => "FIN",
        FR => "FRA",
        GF => "GUF",
        PF => "PYF",
        TF => "ATF",
        GA => "GAB",
        GM => "GMB",
        GE => "GEO",
        DE => "DEU",
        GH => "GHA",
        GI => "GIB",
        GR => "GRC",
        GL => "GRL",
        GD => "GRD",
        GP => "GLP",
        GU => "GUM",
        GT => "GTM",
        GG => "GGY",
        GN => "GIN",
        GW => "GNB",
        GY => "GUY",
        HT => "HTI",
        HM => "HMD",
        VA => "VAT",
        HN => "HND",
        HK => "HKG",
        HU => "HUN",
        IS => "ISL",
        IN => "IND",
        ID => "IDN",
        IR => "IRN",
        IQ => "IRQ",
        IE => "IRL",
        IM => "IMN",
        IL => "ISR",
        IT => "ITA",
        JM => "JAM",
        JP => "JPN",
        JE => "JEY",
        JO => "JOR",
        KZ => "KAZ",
        KE => "KEN",
        KI => "KIR",
        KP => "PRK",
        KR => "KOR",
        KR => "KOR",
        KW => "KWT",
        KG => "KGZ",
        LA => "LAO",
        LV => "LVA",
        LB => "LBN",
        LS => "LSO",
        LR => "LBR",
        LY => "LBY",
        LY => "LBY",
        LI => "LIE",
        LT => "LTU",
        LU => "LUX",
        MO => "MAC",
        MK => "MKD",
        MG => "MDG",
        MW => "MWI",
        MY => "MYS",
        MV => "MDV",
        ML => "MLI",
        MT => "MLT",
        MH => "MHL",
        MQ => "MTQ",
        MR => "MRT",
        MU => "MUS",
        YT => "MYT",
        MX => "MEX",
        FM => "FSM",
        MD => "MDA",
        MC => "MCO",
        MN => "MNG",
        ME => "MNE",
        MS => "MSR",
        MA => "MAR",
        MZ => "MOZ",
        MM => "MMR",
        MM => "MMR",
        NA => "NAM",
        NR => "NRU",
        NP => "NPL",
        NL => "NLD",
        AN => "ANT",
        NC => "NCL",
        NZ => "NZL",
        NI => "NIC",
        NE => "NER",
        NG => "NGA",
        NU => "NIU",
        NF => "NFK",
        MP => "MNP",
        NO => "NOR",
        OM => "OMN",
        PK => "PAK",
        PW => "PLW",
        PS => "PSE",
        PA => "PAN",
        PG => "PNG",
        PY => "PRY",
        PE => "PER",
        PH => "PHL",
        PN => "PCN",
        PL => "POL",
        PT => "PRT",
        PR => "PRI",
        QA => "QAT",
        RE => "REU",
        RO => "ROU",
        RU => "RUS",
        RU => "RUS",
        RW => "RWA",
        SH => "SHN",
        KN => "KNA",
        LC => "LCA",
        PM => "SPM",
        VC => "VCT",
        VC => "VCT",
        VC => "VCT",
        WS => "WSM",
        SM => "SMR",
        ST => "STP",
        SA => "SAU",
        SN => "SEN",
        RS => "SRB",
        SC => "SYC",
        SL => "SLE",
        SG => "SGP",
        SK => "SVK",
        SI => "SVN",
        SB => "SLB",
        SO => "SOM",
        ZA => "ZAF",
        GS => "SGS",
        ES => "ESP",
        LK => "LKA",
        SD => "SDN",
        SR => "SUR",
        SJ => "SJM",
        SZ => "SWZ",
        SE => "SWE",
        CH => "CHE",
        SY => "SYR",
        TW => "TWN",
        TW => "TWN",
        TJ => "TJK",
        TZ => "TZA",
        TH => "THA",
        TL => "TLS",
        TG => "TGO",
        TK => "TKL",
        TO => "TON",
        TT => "TTO",
        TT => "TTO",
        TN => "TUN",
        TR => "TUR",
        TM => "TKM",
        TC => "TCA",
        TV => "TUV",
        UG => "UGA",
        UA => "UKR",
        AE => "ARE",
        GB => "GBR",
        US => "USA",
        UM => "UMI",
        UY => "URY",
        UZ => "UZB",
        VU => "VUT",
        VE => "VEN",
        VE => "VEN",
        VN => "VNM",
        VN => "VNM",
        VG => "VGB",
        VI => "VIR",
        WF => "WLF",
        EH => "ESH",
        YE => "YEM",
        ZM => "ZMB",
        ZW => "ZWE"
    );
    return $codeMap{$code} || $code;
}

1;
