# Plugin for TWiki Enterprise Collaboration Platform, http://TWiki.org/
#
# Copyright (C) 2019-2023 ALPS Aquaservice
# Copyright (C) 2019-2023 Peter Thoeny, peter09[at]thoeny.org
# and TWiki Contributors. All Rights Reserved.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version. For
# more details read LICENSE in the root of this distribution.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details, published at
# http://www.gnu.org/copyleft/gpl.html
#
# As per the GPL, removal of this notice is prohibited.

package TWiki::Plugins::ReplaceTextPlugin::Core;

our $debug = $TWiki::cfg{Plugins}{ReplaceTextPlugin}{Debug} || 0;
our $accessGroup = $TWiki::cfg{Plugins}{ReplaceTextPlugin}{AccessGroup} || '';
our $customModule = $TWiki::cfg{Plugins}{ReplaceTextPlugin}{CustomReplace} || '';
our $customChecked = 0;
our $custom;

# =========================
sub new {
    my ( $class ) = @_;

    my $this = {};
    bless( $this, $class );
    _writeDebug( "new() - constructor" );

    return $this;
}

# =========================
sub VarREPLACETEXT {
    my ( $this, $params, $topic, $theWeb, $meta ) = @_;

    _writeDebug( "VarREPLACETEXT( " . $params->stringify() . " )" );
    my $action = $params->{action};
    if( $action eq 'replace' ) {
        my $web      = $params->{web} || $theWeb;
        my $topic    = $params->{topic} || '';
        my $filter   = $params->{filter} || '';
        my $from     = $params->{from};
        my $to       = $params->{to};
        my $search   = $params->{search} || '\\b$from\\b';
        my $replace  = $params->{replace} || '$to';
        my $dryrun   = $params->{dryrun};
        my $showlist = $params->{showlist};
        return $this->handleReplaceAction( $web, $topic, $filter, $from, $to, $search, $replace, $dryrun, $showlist );

    } elsif( $action eq 'check' ) {
        return 'installed';

    } elsif( $action eq '' ) {
        # return empty string if no action given
        return '';

    } else {
        return " ERROR: Unknown action '$action'";
    }
}

# =========================
sub handleReplaceAction {
    my ( $this, $theWeb, $topic, $filter, $from, $to, $search, $replace, $dryrun, $showlist ) = @_;

    _writeDebug( "handleReplaceAction($theWeb, $topic, $filter, $from, $to, $search, $replace, $dryrun)" ) if( $debug );

    my $query = TWiki::Func::getCgiQuery();
    unless($query) {
        return "REPLACETEXT runs only under CGI web environment";
    }

    unless($theWeb && $from && $to) {
        return "Missing parameters: =web=, =from=, and =to= are required";
    }

    if( $accessGroup && !TWiki::Func::isGroupMember( $accessGroup )) {
        return "Sorry, only members of the $accessGroup can use this replace feature";
    }

    my @topics = TWiki::Func::getTopicList( $theWeb );
    if($topic) {
        $topic = _makeTopicPattern( $topic );
        @topics = grep( /$topic/i, @topics );
    }
    if($filter) {
        my $filtered = TWiki::Func::searchInWebContent( $filter, $theWeb, \@topics,
            { type => 'regex', files_without_match => 1 } );
        @topics = keys %$filtered;
    }
    $search =~ s/\$from/\Q$from\E/;
    $replace =~ s/\$to/$to/;

    if(!$customChecked) {
        $customChecked = 1;
        if($customModule) {
            eval 'require ' . $customModule;
            if( $@ ) {
                _writeDebug( "handleReplaceAction() => error loading module $customModule" );
            } else {
                $custom = $customModule->new(
                    debug => $debug
                );
            }
        }
    }

    my $result = TWiki::Func::searchInWebContent( $search, $theWeb, \@topics,
        { type => 'regex', files_without_match => 1 } );
    my $nTopics = 0;
    @topics = ();
    #my $message = "<verbatim>";
    foreach my $aTopic ( sort keys %$result ) {
        my $text = TWiki::Func::readTopicText($theWeb, $aTopic);
        # check for oops URL in case of error:
        if( $text =~ /^http.*?\/oops/ ) {
            _writeDebug( "handleReplaceAction() => redirect $text" );
            TWiki::Func::redirectCgiQuery( $query, $text );
             return;
        }
        $nTopics++;
        push(@topics, "[[$aTopic]]");
        if($custom) {
            $text = $custom->replaceText( $theWeb, $aTopic, $text, $search, $replace );
        } elsif($replace =~ /\$[1-8]/) {
            # special case: $1, $2 etc are not expanded in $replace (nesting)
            $text =~ s/$search/_replacer($replace, $1, $2, $3, $4, $5, $6, $7, $8 )/ge;
        } else {
            $text =~ s/$search/$replace/g;
        }
        #$message .= "==========\n$text\n==========\n";
        unless($dryrun) {
            TWiki::Func::writeLog( 'replacetext', "$from => $to", $theWeb, $aTopic);
            my $oopsUrl = TWiki::Func::saveTopicText( $theWeb, $aTopic, $text ); # save topic text
            if($oopsUrl) {
                _writeDebug( "handleReplaceAction() => redirect $oopsUrl" );
                TWiki::Func::redirectCgiQuery( $query, $oopsUrl );
                return;
            }
        }
    }
    #$message .= "</verbatim>";
    my $message = $dryrun ? "Simulate replacing " : "Replaced ";
    $message .= "<b>$from</b> with <b>$to</b> in $nTopics topics in $theWeb web";
    if($showlist && $nTopics) {
        $message .= ": " . join(", ", @topics);
    }
    _writeDebug( "handleReplaceAction() => $message" );
    return $message;
}

#==========================
sub _replacer {
    my( $text, $p1, $p2, $p3, $p4, $p5, $p6, $p7, $p8 ) = @_;
    $text =~ s/\$1/$p1/g;
    $text =~ s/\$2/$p2/g;
    $text =~ s/\$3/$p3/g;
    $text =~ s/\$4/$p4/g;
    $text =~ s/\$5/$p5/g;
    $text =~ s/\$6/$p6/g;
    $text =~ s/\$7/$p7/g;
    $text =~ s/\$8/$p8/g;
    return $text;
}

#==========================
sub _makeTopicPattern {
    my ( $topic ) = @_;
    return '' unless ( $topic );

    # 'Web*, FooBar' ==> ( 'Web*', 'FooBar' ) ==> ( 'Web.*', "FooBar" )
    my @arr =
      map { s/[^\*\_\-\+$TWiki::regex{mixedAlphaNum}]//go; s/\*/\.\*/go; $_ }
      split( /,\s*/, $topic );
    return '' unless ( @arr );

    # ( 'Web.*', 'FooBar' ) ==> "^(Web.*|FooBar)$"
    return '^(' . join( '|', @arr ) . ')$';
}

# =========================
sub _writeDebug
{
    my ( $msg ) = @_;
    return unless( $debug );
    TWiki::Func::writeDebug( "- ReplaceTextPlugin::Core::$msg" );
}

# =========================
1;
