#
# Copyright (C) 2000-2001 Andrea Sterbini, a.sterbini@flashnet.it
# Copyright (C) 2001 Peter Thoeny, Peter@Thoeny.com
#
# Authors (in alphabetical order)
#   Andrea Bacchetta
#   Richard Bennett
#   Anthon Pang
#   Andrea Sterbini
#   Martin Watt
#   Rafael Alvarez
#   Thomas Eschner
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details, published at 
# http://www.gnu.org/copyleft/gpl.html
#
# =========================
#
# Manages the cache
#
# =========================
package TWiki::Plugins::Xp::Cache;
use strict;

use TWiki::Plugins::XpTrackerPlugin;
 

my %cachedProjectTeams=();
my %cachedTeamIterations=();
my %cachedIterationStories=();
my $loadedWeb="";

#-----------------------------------------------------------------------------
sub getCacheFileName {
	my $web = shift;
	return "$TWiki::dataDir/$web/.xpcache";
}

#-----------------------------------------------------------------------------
sub buildIterationStoriesCache {
	my $web = shift;
	my @stories = &TWiki::Plugins::XpTrackerPlugin::xpGetAllStories( $web );
    
    
    foreach my $eachS ( @stories ) {
        my $storyText = &TWiki::Store::readTopic($web, $eachS);

        # To go from iteration -> story (multiple values)
        my $iter = TWiki::Plugins::XpTrackerPlugin::xpGetValue("\\*Iteration\\*", $storyText, "storyiter");
        $cachedIterationStories{$iter} .= "$eachS ";
    }	
	return %cachedIterationStories;
}

#-----------------------------------------------------------------------------
sub buildTeamIterationsCache {
	my $web = shift;
	
    foreach my $eachI (keys %cachedIterationStories) {
        my $iterText = &TWiki::Store::readTopic($web, $eachI);

        # To go from team -> iteration (multiple values)
        my $team = TWiki::Plugins::XpTrackerPlugin::xpGetValue("\\*Team\\*", $iterText, "notagsforthis");
        $cachedTeamIterations{$team} .= "$eachI ";

    }
	return %cachedTeamIterations;
}

#-----------------------------------------------------------------------------
sub buildProjectTeamCache {
	my $web = shift;
	
	
	foreach my $eachT (keys %cachedTeamIterations) {
        my $teamText = &TWiki::Store::readTopic($web, $eachT);

        # To go from project -> team (multiple values)
        my $project = TWiki::Plugins::XpTrackerPlugin::xpGetValue("\\*Project\\*", $teamText, "notagsforthis");
        $cachedProjectTeams{$project} .= "$eachT ";
    }
	
	return %cachedProjectTeams;
	
}

#-----------------------------------------------------------------------------
sub buildCache {
	my $web = shift;
    my ($eachP, $eachI, $eachS, $eachT, $allS, $allI);

    # Put the return in here, and suddenly, no caching.
    #return;

    cleanCache();

    # Get all the stories and their iterations:
    buildIterationStoriesCache($web);
    buildTeamIterationsCache($web);
	buildProjectTeamCache($web);
    

    # dump information to disk cache file
    my $projCache = "";
    my $teamCache = "";
    my $iterCache = "";
    my @projects = getAllProjects($web);
    foreach my $project (@projects) {

        my @teams = getProjectTeams($project,$web);
        $projCache .= "PROJ : $project : @teams \n";
        foreach my $team (@teams) {

            my @teamIters = getTeamIterations($team,$web);
            $teamCache .= "TEAM : $team : @teamIters \n";
            foreach my $iter (@teamIters) {

                my @iterStories = getIterStories($iter,$web);
                $iterCache .= "ITER : $iter : @iterStories \n";
            }
        }
    }
	
    my $cacheText = $projCache.$teamCache.$iterCache;
    &TWiki::Store::saveFile(getCacheFileName($web), $cacheText);
}    

#-------------------------------------------------------------------------------
sub getAllProjects {
	return keys %cachedProjectTeams;	
}

#-------------------------------------------------------------------------------
sub getProjectTeams {
	my ($project, $web) = @_;
    return defined($cachedProjectTeams{$project}) ? split( /\s+/, $cachedProjectTeams{$project} ) : ();
	
}

#-------------------------------------------------------------------------------
sub getTeamIterations {
	my ($team, $web) = @_;
    return defined($cachedTeamIterations{$team}) ? split( /\s+/, $cachedTeamIterations{$team} ) : ();
}

#-------------------------------------------------------------------------------
sub getIterStories {
	my ($iteration,$web) = @_;
    return defined($cachedIterationStories{$iteration}) ? split( /\s+/, $cachedIterationStories{$iteration} ) : ();

}

#-------------------------------------------------------------------------------
sub initCache {
	my $web = shift;
	
	my $cacheFileName = &getCacheFileName($web);
    

    # if there is no disk cache file, build one
    if (! (-e "$cacheFileName")) {
       # &TWiki::Func::writeDebug( "NO CACHE, BUILDING DISK CACHE" );
        buildCache($web);
    } else {
        # if cache exists but is not most recent file, rebuild it
        # Do this by checking directory timestamp
        my @cacheStat = stat("$cacheFileName");
        my @latestStat = stat("$TWiki::dataDir/$web");
        # field 9 is the last modified timestamp
        if($cacheStat[9] < $latestStat[9]) {
          # &TWiki::Func::writeDebug( "OLD CACHE $cacheStat[9] $latestStat[9]" );
            buildCache($web);
        } elsif (! $web eq $loadedWeb) {
        	readCache($cacheFileName);
        	$loadedWeb=$web;
        }
    }
    
}

#-------------------------------------------------------------------------------
sub cleanCache {
	# Clean the current cache;
	%cachedProjectTeams=();
	%cachedTeamIterations=();
	%cachedIterationStories=();
}

#-------------------------------------------------------------------------------
sub readCache {
	my $cacheFileName= shift;	     
    my $cacheText = &TWiki::Store::readFile($cacheFileName);
    
	cleanCache();
    
    while($cacheText =~ s/PROJ : (.*?) : (.*?)\n//) {
        $cachedProjectTeams{$1} = "$2";
    }

    while($cacheText =~ s/TEAM : (.*?) : (.*?)\n//) {
        $cachedTeamIterations{$1} = "$2";
    }

    while($cacheText =~ s/ITER : (.*?) : (.*?)\n//) {
        $cachedIterationStories{$1} = "$2";
    }
}
1;